pragma solidity ^0.4.8;


import "./StandardToken.sol";
import "./LimitedTransferToken.sol";

/**
* @title Vested token
* @dev Tokens that can be vested for a group of addresses.
*/
contract VestedToken is StandardToken, LimitedTransferToken {

  struct TokenGrant {
    address granter;
    uint256 value;
    uint64 cliff;
    uint64 vesting;
    uint64 start;
  }

  mapping (address => TokenGrant[]) public grants;

  /**
  * @dev Grant tokens to a specified address
  * @param _to address The address which the tokens will be granted to.
  * @param _value uint256 The amount of tokens to be granted.
  * @param _start uint64 Represents time of the begining of the grant.
  * @param _cliff uint64 Represents the cliff period.
  * @param _vesting uint64 Represents the vesting period.
  */
  function grantVestedTokens(
    address _to,
    uint256 _value,
    uint64 _start,
    uint64 _cliff,
    uint64 _vesting) {

    if (_cliff < _start) {
      throw;
    }
    if (_vesting < _start) {
      throw;
    }
    if (_vesting < _cliff) {
      throw;
    }


    TokenGrant memory grant = TokenGrant(msg.sender, _value, _cliff, _vesting, _start);
    grants[_to].push(grant);

    transfer(_to, _value);
  }


    /**
    * @dev Revoke the grant of tokens of a specifed address.
    * @param _holder address The address which will have its tokens revoked.
    * @param _grantId uint The id of the token grant.
    */
  function revokeTokenGrant(address _holder, uint _grantId) {
    TokenGrant grant = grants[_holder][_grantId];

    if (grant.granter != msg.sender) {
      throw;
    }
    uint256 nonVested = nonVestedTokens(grant, uint64(now));

    // remove grant from array
    delete grants[_holder][_grantId];
    grants[_holder][_grantId] = grants[_holder][grants[_holder].length - 1];
    grants[_holder].length -= 1;

    balances[msg.sender] = balances[msg.sender].add(nonVested);
    balances[_holder] = balances[_holder].sub(nonVested);
    Transfer(_holder, msg.sender, nonVested);
  }

 /**
 * @dev Check the amount of grants that an address has.
 * @param _holder address The holder of the grants.
 * @return A uint representing the total amount of grants.
 */
  function tokenGrantsCount(address _holder) constant returns (uint index) {
    return grants[_holder].length;
  }

  /**
  * @dev Get all information about a specifc grant.
  * @param _holder address The address which will have its tokens revoked.
  * @param _grantId uint The id of the token grant.
  * @return Returns all the values that represent a TokenGrant(address, value,
     start, cliff and vesting) plus the vested value at the current time.
  */
  function tokenGrant(address _holder, uint _grantId) constant returns (address granter, uint256 value, uint256 vested, uint64 start, uint64 cliff, uint64 vesting) {
    TokenGrant grant = grants[_holder][_grantId];

    granter = grant.granter;
    value = grant.value;
    start = grant.start;
    cliff = grant.cliff;
    vesting = grant.vesting;

    vested = vestedTokens(grant, uint64(now));
  }

  /**
  * @dev Get the amount of vested tokens at a specifc time.
  * @param grant TokenGrant The grant to be checked.
  * @param time uint64 The time to be checked
  * @return An uint representing the amount of vested tokens of a specifc grant
  on specifc time.
  */
  function vestedTokens(TokenGrant grant, uint64 time) private constant returns (uint256) {
    return calculateVestedTokens(
      grant.value,
      uint256(time),
      uint256(grant.start),
      uint256(grant.cliff),
      uint256(grant.vesting)
    );
  }

  /**
  * @dev Calculate amount of vested tokens at a specifc time.
  * @param tokens uint256 The amount of tokens grantted.
  * @param time uint64 The time to be checked
  * @param start uint64 A time representing the begining of the grant
  * @param _cliff uint64 Represents the cliff period.
  * @param _vesting uint64 Represents the vesting period.
  * @return An uint representing the amount of vested tokensof a specif grant.
  */
  function calculateVestedTokens(
    uint256 tokens,
    uint256 time,
    uint256 start,
    uint256 cliff,
    uint256 vesting) constant returns (uint256 vestedTokens)
    {

    if (time < cliff) {
      return 0;
    }
    if (time >= vesting) {
      return tokens;
    }

    uint256 cliffTokens = tokens.mul(cliff.sub(start)).div(vesting.sub(start));
    vestedTokens = cliffTokens;

    uint256 vestingTokens = tokens.sub(cliffTokens);

    vestedTokens = vestedTokens.add(vestingTokens.mul(time.sub(cliff)).div(vesting.sub(cliff)));
  }

  /**
  * @dev Calculate the amount of non vested tokens at a specific time.
  * @param grant TokenGrant The grant to be checked.
  * @param time uint64 The time to be checked
  * @return An uint representing the amount of non vested tokens of a specifc grant
  on the passed time frame.
  */
  function nonVestedTokens(TokenGrant grant, uint64 time) private constant returns (uint256) {
    return grant.value.sub(vestedTokens(grant, time));
  }

  /**
  * @dev Calculate the date when the holder can trasfer all its tokens
  * @param holder address The address of the holder
  * @return An uint representing the date of the last transferable tokens.
  */
  function lastTokenIsTransferableDate(address holder) constant public returns (uint64 date) {
    date = uint64(now);
    uint256 grantIndex = grants[holder].length;
    for (uint256 i = 0; i < grantIndex; i++) {
      date = SafeMath.max64(grants[holder][i].vesting, date);
    }
  }

  /**
  * @dev Calculate the total amount of transferable tokens of a holder at a given time
  * @param holder address The address of the holder
  * @param time uint64 The specific time.
  * @return An uint representing a holder's total amount of transferable tokens.
  */
  function transferableTokens(address holder, uint64 time) constant public returns (uint256 nonVested) {
    uint256 grantIndex = grants[holder].length;
    for (uint256 i = 0; i < grantIndex; i++) {
      uint256 current = nonVestedTokens(grants[holder][i], time);
      nonVested = nonVested.add(current);
    }

    return SafeMath.min256(balances[holder].sub(nonVested), super.transferableTokens(holder, time));
  }
}
