---
id: api-stability
title: API Stability
---

On the [OpenZeppelin 2.0 release](https://github.com/OpenZeppelin/openzeppelin-solidity/releases/tag/v2.0.0), we committed ourselves to keeping a stable API. We aim to more precisely define what we understand by _stable_ and _API_ here, so users of the library can understand these guarantees and be confident their project won't break unexpectedly.

In a nutshell, the API being stable means _if your project is working today, it will continue to do so_. New contracts and features will be added in minor releases, but only in a backwards compatible way.

## Versioning scheme
We follow [SemVer](https://semver.org/), which means API breakage may occur between major releases. Read more about the [release schedule](release-schedule) to know how often this happens (not very).

## Solidity functions
While the internal implementation of functions may change, their semantics and signature will remain the same. The domain of their arguments will not be less restrictive (e.g. if transferring a value of 0 is disallowed, it will remain disallowed), nor will general state restrictions be lifted (e.g. `whenPaused` modifiers).

If new functions are added to a contract, it will be in a backwards-compatible way: their usage won't be mandatory, and they won't extend functionality in ways that may foreseeable break an application (e.g. [an `internal` method may be added to make it easier to retrieve information that was already available](https://github.com/OpenZeppelin/openzeppelin-solidity/issues/1512)).

### `internal`
This extends not only to `external` and `public` functions, but also `internal` ones: many OpenZeppelin contracts are meant to be used by inheriting them (e.g. `Pausable`, `PullPayment`, the different `Roles` contracts), and are therefore used by calling these functions. Similarly, since all OpenZeppelin state variables are `private`, they can only be accessed this way (e.g. to create new `ERC20` tokens, instead of manually modifying `totalSupply` and `balances`, `_mint` should be called).

`private` functions have no guarantees on their behavior, usage, or existence.

Finally, sometimes language limitations will force us to e.g. make `internal` a function that should be `private` in order to implement features the way we want to. These cases will be well documented, and the normal stability guarantees won't apply.

## Libraries
Some of our Solidity libraries use `struct`s  to handle internal data that should not be accessed directly (e.g. `Roles`). There's an [open issue](https://github.com/ethereum/solidity/issues/4637) in the Solidity repository requesting a language feature to prevent said access, but it looks like it won't be implemented any time soon. Because of this, we will use leading underscores and mark said `struct`s to make it clear to the user that its contents and layout are _not_ part of the API.

## Events
No events will be removed, and their arguments won't be changed in any way. New events may be added in later versions, and existing events may be emitted under new, reasonable circumstances (e.g. [from 2.1 on, `ERC20` also emits `Approval` on `transferFrom` calls](https://github.com/OpenZeppelin/openzeppelin-solidity/issues/707)).

## Gas costs
While attempts will generally be made to lower the gas costs of working with OpenZeppelin contracts, there are no guarantees regarding this. In particular, users should not assume gas costs will not increase when upgrading library versions.

## Bugfixes
The API stability guarantees may need to be broken in order to fix a bug, and we will do so. This decision won't be made lightly however, and all options will be explored to make the change as non-disruptive as possible. When sufficient, contracts or functions which may result in unsafe behaviour will be deprecated instead of removed (e.g. [#1543](https://github.com/OpenZeppelin/openzeppelin-solidity/pull/1543) and [#1550](https://github.com/OpenZeppelin/openzeppelin-solidity/pull/1550)).

## Solidity compiler version
Starting on version 0.5.0, the Solidity team switched to a faster release cycle, with minor releases every few weeks (v0.5.0 was released on November 2018, and v0.5.5 on March 2019), and major, breaking-change releases every couple months (with v0.6.0 scheduled for late March 2019). Including the compiler version in OpenZeppelin's stability guarantees would therefore force the library to either stick to old compilers, or release frequent major updates simply to keep up with newer Solidity releases.

Because of this, **the minimum required Solidity compiler version is not part of the stability guarantees**, and users may be required to upgrade their compiler when using newer versions of OpenZeppelin. Bugfixes will still be backported to older library releases so that all versions currently in use receive these updates.

You can read more about the rationale behind this, the other options we considered and why we went down this path [here](https://github.com/OpenZeppelin/openzeppelin-solidity/issues/1498#issuecomment-449191611).
